<?php
/*
Plugin Name: Global Text Replacer with Elementor Support, Logging and Flexible Whitespace Matching
Description: پلاگینی برای جایگزینی متن در پست‌ها، متاهای ساده و داده‌های المنتور به همراه ثبت لاگ تغییرات. قابلیت اعمال تغییر در کل سایت یا فقط در یک صفحه خاص با تطبیق انعطاف‌پذیر فاصله‌ها.
Version: 1.4
Author: مصطفی نور 
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // جلوگیری از دسترسی مستقیم
}

class Global_Text_Replacer {

    // نام آپشن برای ذخیره لاگ‌ها
    private $log_option = 'gtr_logs';

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_admin_page' ) );
        add_action( 'admin_post_gtr_replace_text', array( $this, 'handle_text_replace' ) );
    }

    // افزودن صفحه تنظیمات به منوی ادمین
    public function add_admin_page() {
        add_menu_page(
            'Global Text Replacer',
            'Text Replacer',
            'manage_options',
            'global-text-replacer',
            array( $this, 'admin_page_html' ),
            'dashicons-edit',
            80
        );
    }

    // نمایش فرم تنظیمات و لاگ‌ها در پنل مدیریت
    public function admin_page_html() {
        ?>
        <div class="wrap">
            <h1>Global Text Replacer</h1>
            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'gtr_replace_nonce', 'gtr_nonce' ); ?>
                <input type="hidden" name="action" value="gtr_replace_text">
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="search_text">متنی که باید پیدا شود</label>
                        </th>
                        <td>
                            <input type="text" name="search_text" id="search_text" class="regular-text" required>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="replace_text">متن جایگزین</label>
                        </th>
                        <td>
                            <input type="text" name="replace_text" id="replace_text" class="regular-text" required>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">محدوده تغییرات</th>
                        <td>
                            <fieldset>
                                <label>
                                    <input type="radio" name="scope" value="global" checked>
                                    در کل سایت
                                </label><br>
                                <label>
                                    <input type="radio" name="scope" value="page">
                                    فقط در یک صفحه خاص
                                </label>
                            </fieldset>
                        </td>
                    </tr>
                    <tr id="page_url_row" style="display: none;">
                        <th scope="row">
                            <label for="page_url">آدرس صفحه مورد نظر</label>
                        </th>
                        <td>
                            <input type="url" name="page_url" id="page_url" class="regular-text" placeholder="https://example.com/sample-page">
                        </td>
                    </tr>
                </table>
                <?php submit_button( 'جایگزینی متن' ); ?>
            </form>

            <hr>
            <h2>گزارش تغییرات (لاگ)</h2>
            <?php
            $logs = get_option( $this->log_option, array() );
            if ( ! empty( $logs ) ) {
                echo '<table class="widefat fixed striped">';
                echo '<thead><tr><th>تاریخ</th><th>متن جستجو</th><th>متن جایگزین</th><th>تعداد تغییرات</th></tr></thead><tbody>';
                foreach ( $logs as $log ) {
                    echo '<tr>';
                    echo '<td>' . esc_html( $log['date'] ) . '</td>';
                    echo '<td>' . esc_html( $log['search'] ) . '</td>';
                    echo '<td>' . esc_html( $log['replace'] ) . '</td>';
                    echo '<td>' . esc_html( $log['count'] ) . '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';
            } else {
                echo '<p>هنوز تغییراتی ثبت نشده است.</p>';
            }
            ?>
        </div>
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            var radios = document.getElementsByName('scope');
            function togglePageUrl() {
                var selected = document.querySelector('input[name="scope"]:checked').value;
                document.getElementById('page_url_row').style.display = (selected === 'page' ? 'table-row' : 'none');
            }
            for (var i = 0; i < radios.length; i++) {
                radios[i].addEventListener('change', togglePageUrl);
            }
            togglePageUrl();
        });
        </script>
        <?php
    }

    // پردازش فرم و انجام جایگزینی متن
    public function handle_text_replace() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'دسترسی غیرمجاز است.' );
        }

        // اعتبارسنجی nonce
        check_admin_referer( 'gtr_replace_nonce', 'gtr_nonce' );

        // دریافت مقادیر ارسال شده از فرم
        $search_text  = isset( $_POST['search_text'] ) ? sanitize_text_field( wp_unslash( $_POST['search_text'] ) ) : '';
        $replace_text = isset( $_POST['replace_text'] ) ? sanitize_text_field( wp_unslash( $_POST['replace_text'] ) ) : '';
        $scope        = isset( $_POST['scope'] ) ? sanitize_text_field( wp_unslash( $_POST['scope'] ) ) : 'global';

        if ( empty( $search_text ) ) {
            wp_redirect( add_query_arg( 'message', 'error', admin_url( 'admin.php?page=global-text-replacer' ) ) );
            exit;
        }

        // تبدیل رشته جستجو به الگوی regex با تطبیق انعطاف‌پذیر فاصله‌ها
        $pattern = preg_quote( $search_text, '/' );
        $pattern = str_replace( '\ ', '\s*', $pattern );

        $modified_count = 0;

        // در حالت "فقط در یک صفحه خاص"
        if ( $scope === 'page' ) {
            $page_url = isset( $_POST['page_url'] ) ? esc_url_raw( wp_unslash( $_POST['page_url'] ) ) : '';
            if ( empty( $page_url ) ) {
                wp_redirect( add_query_arg( 'message', 'error', admin_url( 'admin.php?page=global-text-replacer' ) ) );
                exit;
            }
            $parsed_url = parse_url( $page_url );
            if ( ! isset( $parsed_url['path'] ) ) {
                wp_redirect( add_query_arg( 'message', 'error', admin_url( 'admin.php?page=global-text-replacer' ) ) );
                exit;
            }
            // استخراج slug از مسیر صفحه
            $slug = trim( $parsed_url['path'], '/' );
            // جستجوی پست/صفحه بر اساس slug
            $post_data = get_page_by_path( $slug, OBJECT, array( 'post', 'page' ) );
            if ( ! $post_data ) {
                wp_redirect( add_query_arg( 'message', 'error', admin_url( 'admin.php?page=global-text-replacer' ) ) );
                exit;
            }
            $posts = array( $post_data );
        } else {
            // دریافت تمامی پست‌ها
            $args  = array(
                'post_type'      => 'any',
                'posts_per_page' => -1,
                'post_status'    => array( 'publish', 'pending', 'draft', 'future', 'private', 'inherit' )
            );
            $posts = get_posts( $args );
        }

        // پردازش پست‌ها برای جایگزینی متن
        foreach ( $posts as $post ) {
            $updated = false;

            // جایگزینی در محتوای اصلی پست
            if ( preg_match( '/' . $pattern . '/u', $post->post_content ) ) {
                $new_content = preg_replace( '/' . $pattern . '/u', $replace_text, $post->post_content );
                wp_update_post( array(
                    'ID'           => $post->ID,
                    'post_content' => $new_content,
                ) );
                $modified_count++;
                $updated = true;
            }

            // بررسی متاهای پست
            $meta_keys = get_post_custom_keys( $post->ID );
            if ( ! empty( $meta_keys ) ) {
                foreach ( $meta_keys as $meta_key ) {
                    $meta_value = get_post_meta( $post->ID, $meta_key, true );
                    if ( is_string( $meta_value ) && preg_match( '/' . $pattern . '/u', $meta_value ) ) {
                        $new_meta_value = preg_replace( '/' . $pattern . '/u', $replace_text, $meta_value );
                        update_post_meta( $post->ID, $meta_key, $new_meta_value );
                        if ( ! $updated ) {
                            $modified_count++;
                            $updated = true;
                        }
                    }
                    // بررسی و جایگزینی در داده‌های المنتور
                    if ( $meta_key === '_elementor_data' && is_string( $meta_value ) ) {
                        $elementor_data = json_decode( $meta_value, true );
                        if ( is_array( $elementor_data ) ) {
                            $new_elementor_data = $this->replace_text_in_elementor_data( $elementor_data, $search_text, $replace_text );
                            if ( json_encode( $elementor_data ) !== json_encode( $new_elementor_data ) ) {
                                update_post_meta( $post->ID, '_elementor_data', wp_slash( json_encode( $new_elementor_data ) ) );
                                if ( ! $updated ) {
                                    $modified_count++;
                                    $updated = true;
                                }
                            }
                        }
                    }
                }
            }
        }

        // ثبت گزارش تغییرات
        $logs = get_option( $this->log_option, array() );
        $logs[] = array(
            'date'    => current_time( 'mysql' ),
            'search'  => $search_text,
            'replace' => $replace_text,
            'count'   => $modified_count
        );
        update_option( $this->log_option, $logs );

        wp_redirect( add_query_arg( 'message', 'success', admin_url( 'admin.php?page=global-text-replacer' ) ) );
        exit;
    }

    // تابع بازگشتی برای جایگزینی متن در آرایه‌های داده (مثلاً در JSON المنتور) با تطبیق انعطاف‌پذیر فاصله‌ها
    private function replace_text_in_elementor_data( $data, $search, $replace ) {
        $pattern = preg_quote( $search, '/' );
        $pattern = str_replace( '\ ', '\s*', $pattern );
        foreach ( $data as $key => $value ) {
            if ( is_array( $value ) ) {
                $data[$key] = $this->replace_text_in_elementor_data( $value, $search, $replace );
            } elseif ( is_string( $value ) ) {
                if ( preg_match( '/' . $pattern . '/u', $value ) ) {
                    $data[$key] = preg_replace( '/' . $pattern . '/u', $replace, $value );
                }
            }
        }
        return $data;
    }
}

new Global_Text_Replacer();
